//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoList: View {
    // START:NewListData
    private enum ListData {
        case list(TodoItemList)
        case items(LocalizedStringKey, [TodoItem])
        // START_HIGHLIGHT
        case group(TodoItemGroup)
        // END_HIGHLIGHT
    }

    // <literal:elide> Properties </literal:elide>
    // END:NewListData
    // START:UpdatedState
    // START_HIGHLIGHT
    @State private var sortBy: SortOption = .manual
    // END_HIGHLIGHT
    @State private var showingChooser: Bool = false
    @Environment(\.presentationMode) private var presentationMode
    // END:UpdatedState
    @EnvironmentObject private var data: DataCenter

    // START:NewItemState
    private static let itemTemplate = TodoItem( //<label id="code.ch5.itemtemplate"/>
        id: Int.min, title: "New Item", priority: .normal,
        notes: nil, date: nil, listID: 2002, complete: false)
    
    @State private var editingItem = Self.itemTemplate
    // END:NewItemState
    
    // START:MultipleSheets
    private enum EditorID: Identifiable, Hashable {
        case itemEditor
        case listEditor
        
        var id: EditorID { self }
    }
    
    @State private var presentedEditor: EditorID? = nil
    // END:MultipleSheets

    // START:DynamicListUpdates
    @State private var listData: ListData
    // END:DynamicListUpdates
    // START:NewListData

    init(list: TodoItemList) {
        self._listData = State(wrappedValue: .list(list))
    }

    init(title: LocalizedStringKey, items: [TodoItem]) {
        self._listData = State(wrappedValue: .items(title, items))
    }

    // START_HIGHLIGHT
    init(group: TodoItemGroup) {
        self._listData = State(wrappedValue: .group(group))
    }
    // END_HIGHLIGHT
    // END:NewListData
    // START:DynamicListUpdates
    // <literal:elide> ... </literal:elide>

    var body: some View {
        List {
            // <literal:elide> ... </literal:elide>
            // END:DynamicListUpdates
            ForEach(sortedItems) { item in
                NavigationLink(destination: TodoItemDetail(item: item)
                    .environmentObject(self.data)
                ) {
                    TodoItemRow(item: item)
                        .accentColor(self.color(for: item))
                }
            }
            // START:DeleteAndMove
            .onDelete { self.removeTodoItems(atOffsets: $0) }
            .onMove(perform: self.sortBy == .manual
                ? { self.moveTodoItems(fromOffsets: $0, to: $1) }
                : nil) // <label id="code.ch6.move.nil" />
            // END:DeleteAndMove
            // START:DynamicListUpdates
        }
        // <literal:elide> existing view modifiers </literal:elide>
        // END:DynamicListUpdates
        .navigationBarTitle(title)
        .navigationBarItems(trailing: barItems)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.title)) {
                        self.sortBy = opt
                    }
            })
        }
        // START:ShowEditors
        .sheet(item: $presentedEditor) { which -> AnyView in
            switch which {
            case .itemEditor:
                return AnyView(
                    self.editorSheet
                        .environmentObject(self.data)
                )
            case .listEditor:
                return AnyView(
                    TodoListEditor(list: self.list!)
                        .environmentObject(self.data)
                )
            }
        }
        // END:ShowEditors
        // START:DynamicListUpdates
        // START_HIGHLIGHT
        .onReceive(data.$todoItems.combineLatest(data.$todoLists)) { _ in
            self.updateData()
        }
        // END_HIGHLIGHT
    }
    // END:DynamicListUpdates
}

// MARK: - Helper Properties

extension TodoList {
    private var sortButton: some View {
        // START:NewSortButtonImage
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Sort List"))
        }
        // END:NewSortButtonImage
    }

    // START:NewTodoItem
    private var addButton: some View {
        Button(action: {
            self.editingItem = Self.itemTemplate
            self.presentedEditor = .itemEditor
        }) {
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New To-Do Item"))
        }
    }

    private var editorSheet: some View {
        let done = Button(action:{
            self.data.addTodoItem(self.editingItem)
            self.presentedEditor = nil
        }) {
            Text("Done")
                .bold()
        }
        let cancel = Button("Cancel") {
            self.presentedEditor = nil
        }
        return NavigationView {
            TodoItemEditor(item: $editingItem)
                .navigationBarItems(leading: cancel, trailing: done)
        }
    }
    // END:NewTodoItem

    // START:NewBarItems
    private var barItems: some View {
        HStack(spacing: 14) {
            if isList {
                // START_HIGHLIGHT
                Button(action: { self.presentedEditor = .listEditor }) {
                    // END_HIGHLIGHT
                    Image(systemName: "info.circle")
                        // START_HIGHLIGHT
                        .imageScale(.large)
                        // END_HIGHLIGHT
                }
            }
            sortButton
            // START_HIGHLIGHT
            addButton
            EditButton()
            // END_HIGHLIGHT
        }
    }
    // END:NewBarItems

    private var isList: Bool {
        if case .list = self.listData {
            return true
        }
        return false
    }
    
    private var list: TodoItemList? {
        if case let .list(list) = self.listData {
            return list
        }
        return nil
    }

    private func forciblyDismiss() {
        presentationMode.wrappedValue.dismiss()
    }

    // START:UnpackGroups
    private var items: [TodoItem] {
        switch listData {
        case .list(let list): return data.items(in: list)
        case .items(_, let items): return items
        // START_HIGHLIGHT
        case .group(let group): return group.items(from: data)
        // END_HIGHLIGHT
        }
    }

    private var title: LocalizedStringKey {
        switch listData {
        case .list(let list): return LocalizedStringKey(list.name)
        case .items(let name, _): return name
        // START_HIGHLIGHT
        case .group(let group): return group.title
        // END_HIGHLIGHT
        }
    }

    private func color(for item: TodoItem) -> Color {
        switch listData {
        case .list(let list): return list.color.uiColor
        case .items: return data.list(for: item).color.uiColor
        // START_HIGHLIGHT
        case .group(let group): return group.color
        // END_HIGHLIGHT
        }
    }
    // END:UnpackGroups
}

// MARK: - Sorting

extension TodoList {
    // START:NewSortedItems
    private var sortedItems: [TodoItem] {
        // START_HIGHLIGHT
        if case .manual = sortBy { return items }
        // END_HIGHLIGHT

        return items.sorted {
            switch sortBy {
            case .title:
                return $0.title.lowercased() < $1.title.lowercased()
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) <
                    ($1.date ?? .distantFuture)
            // START_HIGHLIGHT
            case .manual:
                fatalError("unreachable")
            // END_HIGHLIGHT
            }
        }
    }
    // END:NewSortedItems
}

// MARK: - Model Manipulation

extension TodoList {
    // START:OffsetTranslation
    private var usingUnchangedList: Bool {
        sortBy == .manual
    }

    private func translate(offsets: IndexSet) -> IndexSet {
        guard !usingUnchangedList else { return offsets }
        let items = sortedItems // calculate this just once
        return IndexSet(offsets.map { index in
            data.todoItems.firstIndex { $0.id == items[index].id }!
        })
    }

    private func translate(index: Int) -> Int {
        guard !usingUnchangedList else { return index }
        return data.todoItems.firstIndex { $0.id == sortedItems[index].id }!
    }
    // END:OffsetTranslation

    // START:ModificationMethods
    private func removeTodoItems(atOffsets offsets: IndexSet) {
        let realOffsets = translate(offsets: offsets)
        data.removeTodoItems(atOffsets: realOffsets)
    }

    private func moveTodoItems(fromOffsets offsets: IndexSet, to newIndex: Int) {
        let realOffsets = translate(offsets: offsets)
        let realIndex = translate(index: newIndex)
        data.moveTodoItems(fromOffsets: realOffsets, to: realIndex)
    }
    // END:ModificationMethods

    // START:UpdateList
    private func updateData() {
        switch listData {
        case let .items(title, items): // <label id="code.ch6.update.items"/>
            let newItems = data.items(withIDs: items.map { $0.id })
            listData = .items(title, newItems)

        case let .list(list): // <label id="code.ch6.update.list.instance"/>
            if let newList = data.todoLists.first(where: { $0.id == list.id }) {
                listData = .list(newList)
            }
            else {
                // List is gone!
                forciblyDismiss() // <label id="code.ch6.list.gone" />
            }

        case .group:
            break
        }
    }
    // END:UpdateList
}

// START:ManualSortOption
fileprivate enum SortOption: String, CaseIterable {
    case title = "Title"
    case priority = "Priority"
    case dueDate = "Due Date"
    // START_HIGHLIGHT
    case manual = "Manual"
    // END_HIGHLIGHT

    var title: LocalizedStringKey { LocalizedStringKey(rawValue) }
}
// END:ManualSortOption

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        let data = DataCenter()
        return NavigationView {
            TodoList(title: "All Items", items: data.todoItems)
                .environmentObject(DataCenter())
        }
    }
}
